"use strict";
/**
 * 简化版 Transformer，用于 Node.js 环境的格式化
 * 只做字符串级别的转换，不依赖自定义 Node
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.TABLE_TRANSFORMER = exports.MERMAID_TRANSFORMER = exports.TODO_LIST_TRANSFORMER = void 0;
const table_1 = require("@lexical/table");
const lexical_1 = require("lexical");
/**
 * TodoList Transformer（简化版）
 * 保持 TodoList 格式不变，避免被误解析
 */
exports.TODO_LIST_TRANSFORMER = {
    dependencies: [],
    export: () => null,
    importRegExp: /^## Todo List\n((?:- \[ \] .+\n?)+)/,
    regExp: /^## Todo List\n((?:- \[ \] .+\n?)+)/,
    replace: () => null,
    trigger: '',
    type: 'text-match',
};
/**
 * Mermaid Transformer（简化版）
 * 保持 Mermaid 代码块格式不变
 */
exports.MERMAID_TRANSFORMER = {
    dependencies: [],
    export: () => null,
    importRegExp: /^```mermaid\n/,
    regExp: /^```mermaid\n/,
    replace: () => null,
    trigger: '',
    type: 'text-match',
};
/**
 * Table Transformer
 * 处理 Markdown 表格格式
 */
exports.TABLE_TRANSFORMER = {
    dependencies: [table_1.TableNode, table_1.TableRowNode, table_1.TableCellNode],
    export: (node) => {
        console.log('[TABLE_TRANSFORMER] export called, node type:', node.getType());
        if (!(0, table_1.$isTableNode)(node)) {
            return null;
        }
        const rows = node.getChildren();
        if (rows.length === 0) {
            console.log('[TABLE_TRANSFORMER] export: table has no rows');
            return null;
        }
        console.log('[TABLE_TRANSFORMER] export: table has', rows.length, 'rows');
        const lines = [];
        let isFirstRow = true;
        for (const row of rows) {
            const cells = row.getChildren();
            const cellTexts = cells.map((cell) => {
                return cell.getTextContent().trim();
            });
            lines.push('| ' + cellTexts.join(' | ') + ' |');
            // 在第一行后添加分隔行（与 webview 格式一致）
            if (isFirstRow) {
                const separators = cells.map(() => '---');
                lines.push('| ' + separators.join(' | ') + ' |');
                isFirstRow = false;
            }
        }
        const result = lines.join('\n') + '\n';
        console.log('[TABLE_TRANSFORMER] export result:', result);
        return result;
    },
    regExp: /^\|(.+)\|$/,
    replace: (parentNode, children, match, isImport) => {
        const lineText = match[0];
        // 检查是否是分隔行
        const replacedLineText = lineText.trim().replace(' ', '');
        if (/^\|[-:|\s]+\|$/.test(replacedLineText)) {
            parentNode.remove();
            return;
        }
        // 解析单元格
        const cells = parseTableRow(lineText);
        if (cells.length === 0) {
            return;
        }
        // 检查前一个兄弟节点是否是 TableNode
        const prevSibling = parentNode.getPreviousSibling();
        if (prevSibling && (0, table_1.$isTableNode)(prevSibling)) {
            // 追加行到现有表格
            const dataRow = (0, table_1.$createTableRowNode)();
            for (const cellText of cells) {
                const cell = (0, table_1.$createTableCellNode)(table_1.TableCellHeaderStates.NO_STATUS);
                const paragraph = (0, lexical_1.$createParagraphNode)();
                paragraph.append((0, lexical_1.$createTextNode)(cellText));
                cell.append(paragraph);
                dataRow.append(cell);
            }
            prevSibling.append(dataRow);
            parentNode.remove();
        }
        else {
            // 创建新表格（第一行作为表头）
            const table = (0, table_1.$createTableNode)();
            const headerRow = (0, table_1.$createTableRowNode)();
            for (const cellText of cells) {
                const cell = (0, table_1.$createTableCellNode)(table_1.TableCellHeaderStates.ROW);
                const paragraph = (0, lexical_1.$createParagraphNode)();
                paragraph.append((0, lexical_1.$createTextNode)(cellText));
                cell.append(paragraph);
                headerRow.append(cell);
            }
            table.append(headerRow);
            parentNode.replace(table);
        }
    },
    type: 'element',
};
/**
 * 解析表格行，提取单元格内容
 */
function parseTableRow(line) {
    const trimmed = line.trim();
    if (!trimmed.startsWith('|') || !trimmed.endsWith('|')) {
        return [];
    }
    const content = trimmed.slice(1, -1);
    const cells = content.split('|').map((cell) => cell.trim());
    return cells;
}
//# sourceMappingURL=transformers.js.map