"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PerformanceTimer = void 0;
exports.withTiming = withTiming;
exports.withTimingSync = withTimingSync;
/**
 * 性能计时工具
 * 用于测量各个阶段的耗时
 */
class PerformanceTimer {
    /**
     * 启用/禁用性能计时
     */
    static setEnabled(enabled) {
        this.enabled = enabled;
    }
    /**
     * 开始计时
     */
    static start(label) {
        if (!this.enabled)
            return;
        this.timers.set(label, performance.now());
    }
    /**
     * 结束计时并打印耗时
     */
    static end(label) {
        if (!this.enabled)
            return 0;
        const startTime = this.timers.get(label);
        if (startTime === undefined) {
            console.warn(`[Perf] Timer "${label}" was not started`);
            return 0;
        }
        const duration = performance.now() - startTime;
        this.timers.delete(label);
        console.log(`[Perf] ⏱ ${label}: ${duration.toFixed(2)}ms`);
        return duration;
    }
    /**
     * 标记一个时间点（用于累计测量）
     */
    static mark(label) {
        if (!this.enabled)
            return;
        const marks = this.marks.get(label) || [];
        marks.push(performance.now());
        this.marks.set(label, marks);
    }
    /**
     * 获取两个标记之间的时间差
     */
    static measure(startLabel, endLabel) {
        if (!this.enabled)
            return 0;
        const startMarks = this.marks.get(startLabel);
        const endMarks = this.marks.get(endLabel);
        if (!startMarks?.length || !endMarks?.length) {
            console.warn(`[Perf] Missing marks for "${startLabel}" or "${endLabel}"`);
            return 0;
        }
        const duration = endMarks[endMarks.length - 1] - startMarks[0];
        console.log(`[Perf] ⏱ ${startLabel} → ${endLabel}: ${duration.toFixed(2)}ms`);
        return duration;
    }
    /**
     * 清除所有计时器和标记
     */
    static clear() {
        this.timers.clear();
        this.marks.clear();
    }
    /**
     * 打印性能报告摘要
     */
    static summary(title, durations) {
        if (!this.enabled)
            return;
        console.log(`\n[Perf] ========== ${title} ==========`);
        let total = 0;
        for (const [label, duration] of Object.entries(durations)) {
            console.log(`[Perf]   ${label}: ${duration.toFixed(2)}ms`);
            total += duration;
        }
        console.log(`[Perf]   ────────────────────`);
        console.log(`[Perf]   Total: ${total.toFixed(2)}ms`);
        console.log(`[Perf] ================================\n`);
    }
}
exports.PerformanceTimer = PerformanceTimer;
PerformanceTimer.timers = new Map();
PerformanceTimer.marks = new Map();
PerformanceTimer.enabled = true;
/**
 * 便捷的异步函数计时装饰器
 */
async function withTiming(label, fn) {
    PerformanceTimer.start(label);
    try {
        return await fn();
    }
    finally {
        PerformanceTimer.end(label);
    }
}
/**
 * 同步函数计时
 */
function withTimingSync(label, fn) {
    PerformanceTimer.start(label);
    try {
        return fn();
    }
    finally {
        PerformanceTimer.end(label);
    }
}
//# sourceMappingURL=performance.js.map